package puakma.addin.log;

import java.io.IOException;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;
import java.util.logging.LoggingMXBean;

import puakma.addin.pmaAddIn;
import puakma.addin.pmaAddInStatusLine;

/**
 * 
 * Addin mapping logging from Java logging to the Tornado logging.
 * 
 * <p>
 * The mapping in general is not equal since the meaning of logging in java, and in
 * Tornado differs a little bit, but this is just shot how it should be.
 * 
 * <p>
 * Mapping: <table>
 * <tr>
 * <th>Tornado log level</th>
 * <th>Java minimal log level</th>
 * </tr>
 * <tr>
 * <td>0 - none</td>
 * <td>Nothing</td>
 * </tr>
 * <tr>
 * <td>1 - minimal</td>
 * <td>{@link Level#SEVERE}</td>
 * </tr>
 * <tr>
 * <td>2 - standard</td>
 * <td>{@link Level#WARNING}</td>
 * </tr>
 * <tr>
 * <td>3 - detailed</td>
 * <td>{@link Level#INFO}, {@link Level#CONFIG}</td>
 * </tr>
 * <tr>
 * <td>4 - verbose</td>
 * <td>{@link Level#FINE}, {@link Level#FINER}</td>
 * </tr>
 * <tr>
 * <td>5 - full</td>
 * <td>All levels</td>
 * </tr>
 * </table>
 * 
 * 
 * @author Martin Novak, mn@puakma.net
 */
public class LogAddin extends pmaAddIn
{
  private PuakmaLogHandler puakmaHandler;
  private boolean running;
  private pmaAddInStatusLine statusLine;

  public void pmaAddInMain()
  {
    setAddInName("LOG");
    statusLine = createStatusLine();
    statusLine.setStatus("Starting...");
    
    m_pSystem.doInformation("Starting LogAddin", null, null);
    
    initJavaLogging();
    
    while(!addInShouldQuit()) {
      synchronized(this) {
        try {
          this.wait(1000);
        }
        catch(Exception ex) {
          // IGNORE...
        }
        running = true;
      }
    }
    
    // CLEANUP...
    statusLine.setStatus("Shutting down...");
    synchronized(this) {
      running = false;
      returnJavaLogging();
    }
    
    removeStatusLine(statusLine);
    statusLine = null;
  }
  
  public boolean addInReady()
  {
    synchronized(this) {
      return running;
    }
  }

  /**
   * This function initializes java logging.
   */
  private void initJavaLogging()
  {
    Logger topLogger = Logger.getLogger("");
    
    synchronized(topLogger) {
      // REMOVE ALL EXISTING HANDLERS
      Handler[] handlers = topLogger.getHandlers();
      for(int i = 0; i < handlers.length; ++i) {
        topLogger.removeHandler(handlers[i]);
      }
      
      // AND ADD OUR HANDLER
      puakmaHandler = new PuakmaLogHandler(m_pSystem);
      puakmaHandler.setFilter(new TornadoLogFilter(m_pSystem));
      topLogger.addHandler(puakmaHandler);
    }
  }
  
  /**
   * Returns java logging back to the state before this addin took control over logging.
   */
  private void returnJavaLogging()
  {
    LogManager manager = LogManager.getLogManager();
    
    try {
      manager.readConfiguration();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }
}
